#    flameGraphRenderer.py - visualizes time span data in the form of a blame graph (refined flame graph)
#    Copyright (C) 2012, 2013  Frank Brueseke
#
#    This program is free software: you can redistribute it and/or modify
#    it under the terms of the GNU Lesser General Public License as published by
#    the Free Software Foundation, either version 3 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU Lesser General Public License for more details.
#
#    You should have received a copy of the GNU Lesser General Public License
#    along with this program.  If not, see <http://www.gnu.org/licenses/>.


#################################################
# SVG constants:
extraWidth  = 200
rowHeight   = 20
levelOffset = 10
minWidth    = 2
colCycle    = {
	"red"   :("#FF0000","#D40000","#FF2A2A","#AA0000"),
	"green" :("#00FF00","#00D400","#2AFF2A","#00AA00"),
	"yellow":("#FFFF00","#D4D400","#FFFF2A","#AAAA00")
}

#################################################
# functions:

def cutCall(call):
	return call[:call.find("(")]

def keyIsTraceLen(obj):
	return len(obj["trace"])

def getCallInTree(tree, call):
	if tree["call"]==call:
		return tree
	else: 
		for kid in tree["kids"]:
			tmpRes = getCallInTree(kid,call)
			if tmpRes!=None:
				return tmpRes
		return None

def insertCallChainInRoot(tree, callChain):
	call = cutCall(callChain[0])
	if call != tree["call"]:    # root case
		kid = {"call":call, "kids":[]}
		tree["kids"].append(kid)
		return insertCallChainInRoot(kid, callChain)
	else:
		if len(callChain)==1:
			return tree
		else:
			call = cutCall(callChain[1])
			try:
				cIdx = [x["call"] for x in tree["kids"]].index(call)
				return insertCallChainInRoot(tree["kids"][cIdx], callChain[1:])
			except:
				kid = {"call":call, "kids":[]}
				tree["kids"].append(kid)
				return insertCallChainInRoot(kid, callChain[1:])

def readFile(in_file, in_traceCol, in_widthCol, in_blameCol, in_nonBlameCol):
	allNodes = []
	maxWidth = 1
	maxLevel = 0

	fobj = open(in_file, "r")
	for line in fobj:
		splitted = line.split(";")
		traceLine = splitted[in_traceCol].split(',')
		traceLine.reverse()
		splitDict = {"trace"        : traceLine
				, "width"   :float(splitted[in_widthCol])
				, "doBlame" :bool(splitted[in_blameCol].lower()=="true")
				, "noBlame" :bool(splitted[in_nonBlameCol].lower()=="true")
		}
		if splitDict["width"] > maxWidth:
			maxWidth = splitDict["width"]
		if len(traceLine) > maxLevel:
			maxLevel = len(traceLine)
		allNodes.append(splitDict)
	
	allNodes.sort(key=keyIsTraceLen)
	# raise Exception(allNodes)
	# raise Exception([len(obj["rec"]["trace"]) for obj in allNodes])
	return (allNodes, maxWidth, maxLevel)

def buildDataStructure(in_allNodes):
	callTree={"call":"/", "kids":[]}
	for node in in_allNodes:
#		print "#"*40
#		print len(node["trace"])
		curr = callTree
		found = getCallInTree(callTree, cutCall(node["trace"][0]))
		if found:
			curr=found
		insert = insertCallChainInRoot(curr, node["trace"])
		insert["width"]    =node["width"]
		insert["doBlame"]  =node["doBlame"]
		insert["noBlame"]  =node["noBlame"]
#		print insert["call"]
#		print callTree
	
#	print
#	print "#"*40
#	print "#"*40
#	print callTree
	return callTree

def __traverse2FindWidth(tree):
	# traverse from here and find max on path and add up time for neighbors
	resultWidth = 0
	if tree["call"]!="/":
		for kid in tree["kids"]:
			resultWidth += __traverse2FindWidth(kid)
		resultWidth = max(resultWidth, tree.get("width", 0))
	return resultWidth

def traverse2FixWidth(tree):
	# normal traverse on finding missing width:
	# - traverse from here and find max on path and add up time for neighbors
	if tree["call"]!="/" and not "width" in tree:
		tree["widthFix"] = __traverse2FindWidth(tree)
	
	for kid in tree["kids"]:
		traverse2FixWidth(kid)

def getWidthAtFirstLevel(tree):
	result = 0
	for kid in tree["kids"]:
		result += kid.get("widthFix", 0) + kid.get("width", 0)
	return result

def traverseSVG(tree, rects, rx, level, maxLevel, scaleBy, in_width=0, doGrayBox=True):
	rwidth = max(minWidth, in_width, tree.get("widthFix", 0)*scaleBy)
	nextLevel = level+1
	if tree["call"]!="/":    #ignore default root
		ry=(maxLevel+levelOffset-2-level)*rowHeight
		rcolor="silver"
		callTime="n. a."
		sign=""
		svgClazz = "myDef"
		if "width" in tree:
			rwidth = max(minWidth, in_width, tree["width"]*scaleBy)
			callTime = "%.2fms" % max(0,tree["width"])
			if tree["noBlame"] and not tree["doBlame"]:
				svgClazz = "myOK"
				sign="[OK]"
				rcolor = colCycle["green"][len(rects)%len(colCycle)]
			elif not tree["noBlame"] and tree["doBlame"]:
				svgClazz = "myNOK"
				sign="[!]"
				rcolor = colCycle["red"][len(rects)%len(colCycle)]
			else:
				svgClazz = "myDUNNO"
				sign="[?]"
				rcolor = colCycle["yellow"][len(rects)%len(colCycle)]    # HSL-colors were so nice :(
		rDescription="%s(..) %s %s" % (tree["call"], callTime, sign)
		descShortIndex = tree["call"].rfind(".", 0, tree["call"].rfind("."))+1
		rShortDesc  ="%s(..) %s %s" % (tree["call"][descShortIndex:], callTime, sign)
		
		rect = """<rect x="%(rX)f" y="%(rY)f" width="%(width)f" height="%(height)f" fill="%(color)s" rx="2" ry="2" class="%(clazz)s" onmouseover="s('%(description)s')" onmouseout="c()" />
<text text-anchor="" x="%(tX)f" y="%(tY)f" font-size="12" font-family="Verdana" fill="rgb(0,0,0)" onmouseover="s('%(description)s')" onmouseout="c()" >%(shortDesc)s</text>""" % {
			"rX": rx,
			"rY": ry,
			"width": rwidth,
			"height": rowHeight,
			"color": rcolor,
			"description": rDescription,
			"tX": rx+3,
			"tY": ry+rowHeight-3,
			"shortDesc": rShortDesc,
			"clazz": svgClazz
		}
		if ( "width" in tree) or (not "width" in tree and doGrayBox):
			rects.append(rect)
		else:
			nextLevel = level
	startx = rx
	for kid in [x for x in tree["kids"] if "width" in x or "widthFix" in x]:
		rx += traverseSVG(kid, rects, rx, nextLevel, maxLevel, scaleBy, doGrayBox=doGrayBox)
	emptyList  = [x for x in tree["kids"] if not "width" in x and not "widthFix" in x]
	emptyWidth = (rwidth-rx+startx)/max(1,len(emptyList))
	for kid in emptyList:
		rx += traverseSVG(kid, rects, rx, nextLevel, maxLevel, scaleBy, in_width=emptyWidth, doGrayBox=doGrayBox)
	return rwidth

def generateSVG(in_svgFileName, in_svgWidth, in_callTree, maxWidth, maxLevel, doGrayBox):
	scaleFactor = 1
	if in_svgWidth > 0:
		scaleFactor = (in_svgWidth-extraWidth)/(maxWidth)
		maxWidth=in_svgWidth-extraWidth

	#global
	rects=[]
	traverseSVG(in_callTree, rects, 0, 3, maxLevel, scaleFactor, doGrayBox=doGrayBox)

	svgText = """<?xml version="1.0" standalone="no"?>
<!DOCTYPE svg PUBLIC "-//W3C//DTD SVG 1.1//EN" "http://www.w3.org/Graphics/SVG/1.1/DTD/svg11.dtd">
<svg version="1.1" width="%(width)i" height="%(height)i" onload="init(evt)" viewBox="0 0 %(width)i %(height)i" xmlns="http://www.w3.org/2000/svg" >
<defs >
	<linearGradient id="background" y1="0" y2="1" x1="0" x2="0" >
		<stop stop-color="#eeeeee" offset="5%%" />
		<stop stop-color="#eeeeb0" offset="95%%" />
	</linearGradient>
    <marker id="arrowLeft"
      viewBox="0 0 10 10" refX="5" refY="5" 
      markerUnits="strokeWidth" markerWidth="3" markerHeight="10"
      orient="auto">
      <path d="M 10 0 L 0 5 L 10 10 z" fill="silver" stroke="grey"/>
    </marker>
</defs>
<style type="text/css">
	.myHighlight { stroke:blue; stroke-width:3; }
	rect[rx]:hover { stroke:black; stroke-width:2; }
	rect{ stroke:white; stroke-width: 0.5; }
	/*text:hover { stroke:black; stroke-width:1; stroke-opacity:0.35; }*/
	text:hover { font-weight: bold; z-index: 1000; }
	text{ z-index: 20; }
</style>

<script type="text/ecmascript">
<![CDATA[
	var details;
	function init(evt) { details = document.getElementById("details").firstChild; }
	function s(info) { details.nodeValue = info; }
	function c() { details.nodeValue = ' '; }
	
	function showHighlight(clazzName){
		var elems = document.getElementsByClassName(clazzName);
		var elemsLen = elems.length;
		for (var i = 0; i < elemsLen; ++i){
			var elem = elems[i];
			elem.classList.add("myHighlight");
		}
	}
	function clearHighlight(clazzName){
		var elems = document.getElementsByClassName(clazzName);
		var elemsLen = elems.length;
		for (var i = 0; i < elemsLen; ++i){
			var elem = elems[i];
			elem.classList.remove("myHighlight");
		}
	}
]]>
</script>
<rect x="0.0" y="0" width="%(width)i" height="%(height)i" fill="url(#background)"  />
<text text-anchor="middle" x="%(titleX)i" y="24" font-size="17" font-family="Verdana" fill="rgb(0,0,0)"  >Blame Graph</text>
<text text-anchor="left" x="10" y="%(fY)i" font-size="12" font-family="Verdana" fill="rgb(0,0,0)"  >Function:</text>
<text text-anchor="" x="70" y="%(fY)i" font-size="12" font-family="Verdana" fill="rgb(0,0,0)" id="details" > </text>

<path d="M 5 %(legendY)i L %(legendXM)i %(legendY)i"
      stroke="silver" stroke-width="3" fill="none"
      marker-start="url(#arrowLeft)"
/>
<path d="M %(legendXE)i %(legendY)i L %(legendXM)i %(legendY)i"
      stroke="silver" stroke-width="3" fill="none"
      marker-start="url(#arrowLeft)"
/>
<text x="%(legendXM)i" y="%(legendYT)i" text-anchor="middle" font-size="12" font-family="Verdana" fill="dimgrey">box length - median response time (ms)</text>

<rect x="%(legendB1X)f" y="%(legendBY)f" width="120" height="20" fill="lime" rx="2" ry="2" class="legend" onmouseover="showHighlight('myOK')" onmouseout="clearHighlight('myOK')"/>
<text x="%(legendB1X)f" y="%(legendBYT)f" text-anchor="left" font-size="10" font-family="Verdana" fill="black" onmouseover="showHighlight('myOK')" onmouseout="clearHighlight('myOK')">
      &#160; OK calls</text>
<rect x="%(legendB2X)f" y="%(legendBY)f" width="120" height="20" fill="yellow" rx="2" ry="2" class="legend" onmouseover="showHighlight('myDUNNO')" onmouseout="clearHighlight('myDUNNO')"/>
<text x="%(legendB2X)f" y="%(legendBYT)f" text-anchor="left" font-size="10" font-family="Verdana" fill="black" onmouseover="showHighlight('myDUNNO')" onmouseout="clearHighlight('myDUNNO')">
      &#160; undecided calls</text>
<rect x="%(legendB3X)f" y="%(legendBY)f" width="120" height="20" fill="red" rx="2" ry="2" class="legend" onmouseover="showHighlight('myNOK')" onmouseout="clearHighlight('myNOK')"/>
<text x="%(legendB3X)f" y="%(legendBYT)f" text-anchor="left" font-size="10" font-family="Verdana" fill="black" onmouseover="showHighlight('myNOK')" onmouseout="clearHighlight('myNOK')">
      &#160; problematic calls</text>

%(svgRects)s
</svg>""" % {
		"width"   : maxWidth + extraWidth, 
		"height"  : (maxLevel+levelOffset)*rowHeight,
		"fY"      : (maxLevel+levelOffset-1)*rowHeight,
		"titleX"  : min(600, (maxWidth + extraWidth)/2),
		"legendXE": maxWidth-5,
		"legendXM": maxWidth/2,
		"legendY" : (maxLevel+levelOffset-4)*rowHeight,
		"legendYT": (maxLevel+levelOffset-4)*rowHeight+12,
		"legendB1X": maxWidth/2 - 150,
		"legendB2X": maxWidth/2,
		"legendB3X": maxWidth/2 + 150,
		"legendBY": (maxLevel+levelOffset-4)*rowHeight+20,
		"legendBYT": (maxLevel+levelOffset-4)*rowHeight+32,
		"svgRects": "\n".join(rects),
	}

	fobj = open(in_svgFileName, "w")
	print >> fobj,svgText
	fobj.close()

def doGraph(inputFile, traceCol, widthCol, blameCol, nonBlameCol, svgWidth, suffix="", doGrayBox=True):
	svgFileName = inputFile+suffix+".svg"
	print "using svg file "+svgFileName
	print

	fullList, maxWidth, maxLevel = readFile(inputFile, traceCol, widthCol, blameCol, nonBlameCol)
	fullTree = buildDataStructure(fullList)
	traverse2FixWidth(fullTree)
	maxWidth = max(getWidthAtFirstLevel(fullTree), maxWidth)
	#maxWidth = getWidthAtFirstLevel(fullTree)
	generateSVG(svgFileName, svgWidth, fullTree, maxWidth, maxLevel, doGrayBox)

#################################################

def main():
	print "Good day!"
	from optparse import OptionParser
	parser = OptionParser()
	parser.add_option("-f", "--file", dest="inputFile",
        	          help="CSV file that we build a flame graph of.", metavar="FILE")
	parser.add_option("-t", "--trace-col", dest="traceCol", type="int", default=0, 
        	          help="Number of column that holds the stacktrace information (0-based)")
	parser.add_option("-w", "--width-col", dest="widthCol", type="int", default=1, 
        	          help="Number of column that holds the value inteded as bar width in the flame graph (0-based)")
	parser.add_option("-b", "--blame-col", dest="blameCol", type="int", default=3, 
        	          help="Number of column that holds the boolean value indicating if the method shall be blamed (0-based)")
	parser.add_option("-n", "--non-blame-col", dest="nonBlameCol", type="int", default=3, 
        	          help="Number of column that holds the boolean value indicating if the method shall NOT be blamed (0-based)")
	parser.add_option("-s", "--svg-width", dest="svgWidth", type="int", default=0, 
			help="The output SVG graphics will be this many pixels wide (scales picture)(default: 0 for 1px = 1ms)")
	(options, args) = parser.parse_args()

	print "using input file "+options.inputFile
	doGraph(options.inputFile, options.traceCol, options.widthCol, options.blameCol, options.nonBlameCol, options.svgWidth)

if __name__ == "__main__":
    main()


