package de.dsor.ontooptmod.derivation;

import java.util.Collection;
import java.util.HashSet;
import java.util.Hashtable;
import java.util.LinkedList;
import java.util.List;
import java.util.Set;

import org.jdom2.Document;
import org.jdom2.Element;

public class ModelQueryIndividualsStructure implements IModelQueryIndividualsStructure{
	
	//Storing Data of SPARQL-DL Query Results 
	private Document doc;
	private Boolean nonInstantiableSet = false;
	private List<SPARQLQueryResult> results;
	
	/**
	 * Constructor stores the document and calls helper method to load all XML-result tags and their values into a respective Data-Structure.
	 * @param doc
	 */
	ModelQueryIndividualsStructure(Document doc){
	
	//Save the document
	this.doc = doc;	
	
	//Create the List of Results and load them
	this.results = new LinkedList<SPARQLQueryResult>();
	this.loadResults();
	
	}
	
	/**
	 * Helper Method loads the document into a list of result objects
	 */
	private void loadResults(){
		
		//Process Document by respective methods. Avoid Usage of XPath. 
		List<Element> resultslist = this.doc.getRootElement().getChildren("results");
		
		for(Element resultblock:resultslist){
			List<Element> resultlist = resultblock.getChildren("result");
			
			for(Element res:resultlist){
				
				//Process current result block into a result object
				SPARQLQueryResult currentresult = new SPARQLQueryResult();
				List<Element> bindings = res.getChildren();
				for(Element bin:bindings){
					String varname = bin.getAttributeValue("name");
					String textcontent = "MISTAKE";
					
					if(bin.getChild("literal") != null){
						textcontent = bin.getChildText("literal");
					}
					else if(bin.getChild("uri") != null){
						textcontent = bin.getChildText("uri");
					}
					currentresult.addBinding(varname, textcontent);
				}
				//Store the result Object into the results-list
				this.results.add(currentresult);
			}
			
		}
	}
	
	@Override 
	public String toString(){
		String result = "";
		
		for(SPARQLQueryResult res:this.results){
			result += res.toString() + " ";
		}
		return result;
	}
	
	/**
	 * Return the results
	 * @return
	 */
	public List<SPARQLQueryResult> getResults(){
		return this.results;
	}
	
	/**
	 * Method returns the entity iris (as strings) and ENames for the respective variable names that exist in the results. 
	 * Methods implementing concrete Statement Derivations for a Semantic Type can hence call this method with the respective variable names and obtain the respective
	 * information and use it in comparison with the results.
	 * @param varnames
	 * @return
	 */
	public Hashtable<String, Set<String>> getEntitiesandNamesforVariables(List<String> varnames){
		//TODO
		Hashtable<String, Set<String>> entitiesAndNames= new Hashtable<String, Set<String>>();
		
		for(String varname:varnames){
			Set<String> resultsforvarname = new HashSet<String>();
			
			for(SPARQLQueryResult res:this.results){
				String value = res.getValue(varname);
				if(value != null){
					resultsforvarname.add(value);
				}
			}
			entitiesAndNames.put(varname, resultsforvarname);	
		}
		
		return entitiesAndNames;
	}
	
	/**
	 * This method returns all unionset-operands for the entity identified with OM#EName unionindname. The query variable names for the possible unionset and it's operands have 
	 * to be provided as parameters. Result can be tested for being a unionset with the List.isEmpty() Method.
	 * @param unionindname
	 * @param unionvarname
	 * @param operatorvarname
	 */
	public List<String> getUnionSetOperandsinCaseOfUnionSet(String unionindname, String unionvarname, String operatorvarname){
		
		List<String> operandnames = new LinkedList<String>();
		
		for(SPARQLQueryResult res:this.results){
			String actoperand = res.getOperandforUnionSetTypingofEntitybyNameandVar(unionindname, unionvarname, operatorvarname);
			//System.out.println("Here is a UnionSet-Operand:" + actoperand);
			if(actoperand != null){
				operandnames.add(actoperand);
			}
		}
	return operandnames;
	}
	
	/**
	 * This method returns the String that is the ename for a model entitiy that is referenced, e.g., as a definedOn Index or a required class by the the model entity with ename entname
	 * The query variable name for the related element has to be provided!
	 * Before using these as set-index names, checks, e.g., for unionsets will have to be performed.
	 * Attention: This method assumes uniqueness of the element whose name is returned. It will finish with the first element found!
	 * For a method not assuming uniqueness call returnMultipleRelatedEntitiesNamesforEntityName
	 * @param entname
	 * @param relentityvarname
	 * @return
	 */
	public String returnRelatedEntityNameforEntityName(String entname, String relentityvarname){
		
		for(SPARQLQueryResult res:this.results){
			String relentityforvarname = res.getRelatedEntityforEntityName(entname, relentityvarname);
			if(relentityforvarname != null){
				return relentityforvarname;
			}
		}
		return null;
	}
	
public Collection<String> returnMultipleRelatedEntitiesNamesforEntityName(String entname, String relentityvarname){
	    Collection<String> results = new LinkedList<String>();
	
		for(SPARQLQueryResult res:this.results){
			String relentityforvarname = res.getRelatedEntityforEntityName(entname, relentityvarname);
			if(relentityforvarname != null){
				results.add(relentityforvarname);
			}
		}
		return results;
	}
	
	/**
	 * Check all result-blocks for nonInstantiability Information. 
	 */
	public Boolean containsNonInstantiabilityTrueInformation(String entname, String entvarname, String noninstvarname){

		for(SPARQLQueryResult res:this.results){
			boolean info = res.containsNonInstantiabilityTrueInformation(entname, entvarname, noninstvarname);
			System.out.println("Checking nonInstantiability Information for current result block of " + entname + " :" + info);
			if (info == true){return true;}
			}
		return false;
	}

	public Boolean containsVariableTypingforEntity(String entname, String varname){
		for(SPARQLQueryResult res:this.results){
			boolean info = res.constainsVariableTypingforEntity(entname, varname);
			if (info == true){return true;}
			}
		return false;
	}
	
	public Boolean containsOntologyTypeInformationforEntityName(String entname, String entvarname, String ontotypeiri){
		for(SPARQLQueryResult res:this.results){
			boolean info = res.containsOntologyTypeInformationforEntity(entname,entvarname,ontotypeiri);
			if (info == true){return true;}
			}
		return false;
	}
	
	/**
	 * Checks all results blocks wether the mathematical property mprop has been defined for the entity with EName entname
	 * @param entname
	 * @param entvarname
	 * @param mprop
	 * @return
	 */
	public Boolean containsMPropforEntity(String entname, String entvarname, MProp mprop){
		//TODO: Are these checks really entity independent
		switch(mprop){
		case LBZeroEQ: for(SPARQLQueryResult res:this.results){
			boolean info = res.containsNonNegativeLowerBoundMathPropforEntity(entname,entvarname);
			if (info == true){return true;}
			}
			return false;
		case LBZeroSPos: for(SPARQLQueryResult res:this.results){
			boolean info = res.containsStrictlyPositiveLowerBoundMathPropforEntity(entname,entvarname);
			if (info == true){return true;}
			}
			return false;
		case IntegerProp: for(SPARQLQueryResult res:this.results){
			boolean info = res.containsIntegerDataMathPropforEntity(entname,entvarname);
			if (info == true){return true;}
			}
			return false;
		case BinaryProp:for(SPARQLQueryResult res:this.results){
			boolean info = res.containsBinaryDataMathPropforEntity(entname,entvarname);
			if (info == true){return true;}
			}
			return false;
		default: System.err.println("Mistake when switching over mprops to check definitions");
		 	return false;	
		 	
		}
		
		
	}
}
