package de.dsor.ontooptmod.derivation;

import java.util.Hashtable;

/**
 * This class stores SPARQL Query Results and provides access to it.
 * @author Florian Stapel
 *
 */
public class SPARQLQueryResult {
	
	//We store the bindings of a result block in a Hashtable. The keys are the variable names as strings as they are
	//unique for a query result. The values are uris/iris or literals as strings
	private Hashtable<String, String> bindings;
	
	SPARQLQueryResult(){
		this.bindings = new Hashtable<String,String>();
	}
	
	@Override 
	public String toString(){
		return this.bindings.toString();
	}
	
	public void addBinding(String key, String value){
		this.bindings.put(key, value);
	}
	
	public String getValue(String key){
		return this.bindings.get(key);
	}
	
	//Domain specific Information for searching the Results. The following methods provide ontology-optimization model domain-specific information on the content of the
	//SPARQL-DL Query Results.
	
	/*
	 * This method returns true iff there is a tag for a noninstantiablity-query variable that evaluates to true for the variable varname in the result block 
	 * for the individual with EName entname and query variable entvarname.
	 */
	public Boolean containsNonInstantiabilityTrueInformation(String entname, String entvarname, String noninstvarname){
		
		System.out.println("Check that result block contains " + entvarname + ":" + this.bindings.containsKey(entvarname));
		System.out.println("Here come the bindings" + this.bindings);
		System.out.println("Check for matching with entname:" + this.bindings.get(entvarname).equals(entname));
		if(this.bindings.containsKey(entvarname) && (this.bindings.get(entvarname).equals(entname))){
		
				String value = this.bindings.get(noninstvarname);
				System.out.println("I am in the inner result block and trying to find noninstantiability information for " + entname);
				if(value != null){
					return (value.equals("true") || value.equals("xsd:true"));
				}
		}
		return false;
	}
	
	/**
	 * This method returns true iff the result block represented by the current instance of this class
	 * contains a definition of the model entity identified by the Ename entid, retrieved as a result of the variable varname, as a OM#Variable
	 * Assumption: We assume that entname is the EName of an entity to be instantiated. This Ename is retrieved by the variable varname. 
	 * If entid just returns to an individual that occurs in an indexing, and another model entity is typed as a variable in the block, the method will return true!
	 * We check this and then look wether there is a variable definition for the entity in the block 
	 * @param entname
	 * @param varname
	 * @return
	 */
	public Boolean constainsVariableTypingforEntity(String entname, String varname){
	
		//First check wether the pair of EName and Variable matches
		if(this.bindings.containsKey(varname) && (this.bindings.get(varname).equals(entname))){
			//Now check for the variable definition
			return this.bindings.containsValue("http://www.semanticweb.org/florianstapel/ontologies/2014/8/OM.owl#Variable");
		}
		return false;
	}
	
	/**
	 * This method returns true iff the result block represented by the current instance of this class
	 * contains a definition of the model entity identified by the Ename entid, retrieved as a result of the variable varname, as a OM#Variable
	 * Assumption: We assume that entname is the EName of an entity to be instantiated. This Ename is retrieved by the variable varname. 
	 * If entid just returns to an individual that occurs in an indexing, and another model entity is typed as a variable in the block, the method will return true!
	 * We check this and then look wether there is a variable definition for the entity in the block 
	 * @param entname
	 * @param varname
	 * @return
	 */
	public Boolean constainsParameterTypingforEntity(String entname, String varname){
		
		//First check whether the pair of EName and Variable matches
		if(this.bindings.containsKey(varname) && (this.bindings.get(varname).equals(entname))){
			//Now check for the variable definition
			return this.bindings.containsValue("http://www.semanticweb.org/florianstapel/ontologies/2014/8/OM.owl#Parameter");
		}
		return false;
	}
	
	/**
	 * This method returns the EName of an operand, iff unionindname is the OM#EName of a model individual that is being implicitely typed as a unionset in the result block
	 * represented by the current instance. For correctness, ename has to be validated against unionvarname. This Ename is retrieved by the variable unionvarname.
	 * operatorvarname denotes the query variable for unionset-operands
	 * It can be used in a disjunction over all result blocks to find out about all unionset operands
	 * @param unionindname
	 * @return
	 */
	public String getOperandforUnionSetTypingofEntitybyNameandVar(String unionindname, String unionvarname, String operatorvarname){
		
		//First check whether the pair of EName and Variable matches
		//System.out.println("Checking matching of " + unionvarname + " and " + unionindname + " in the current result block");
		//System.out.println("Here come the bindings:" + this.bindings.toString());
		//System.out.println("Contains test:" + this.bindings.containsKey(unionvarname));
		//System.out.println("Equality test:" + (this.bindings.get(unionvarname).equals(unionindname)));
		if( this.bindings.containsKey(unionvarname) && (this.bindings.get(unionvarname).equals(unionindname))){
			//System.out.println("Going to return an indexname for the variable:" + this.bindings.get(operatorvarname));
			//Now check for the existence of operands
			return this.bindings.get(operatorvarname);
		}
		return null;
	}
	

	//MathProps: Individual Methods for the respective Properties
	//TODO: Implementations for generic definitions such as general bounds
	//REMARK: ALL OF THESE CAN BE CONCLUDED FROM THE PRESENCE OF OM.Mathprop individuals in the blocks!

	/**
	 * This method returns true iff the result block represented by the current instance of this class
	 * contains a NonNegativeLowerBound MathProp-Definition. Callers should now the context in order to conclude the right model entity with this property
	 * @param entid
	 * @return
	 */
	public Boolean containsNonNegativeLowerBoundMathPropforEntity(String entname, String entvarname){
		if( this.bindings.containsKey(entvarname) && (this.bindings.get(entvarname).equals(entname))){
			return this.bindings.containsValue("http://www.semanticweb.org/florianstapel/ontologies/2015/6/OM.MProps.owl#ZeroLowerBoundInd");
		}
		return false;
	}
	
	/**
	 * This method returns true iff the result block represented by the current instance of this class
	 * contains a StrictlyPositiveLowerBound MathProp-Definition. Callers should now the context in order to conclude the right model entity with this property
	 * @param entid
	 * @return
	 */
	public Boolean containsStrictlyPositiveLowerBoundMathPropforEntity(String entname, String entvarname){
		if( this.bindings.containsKey(entvarname) && (this.bindings.get(entvarname).equals(entname))){
			return this.bindings.containsValue("http://www.semanticweb.org/florianstapel/ontologies/2015/6/OM.MProps.owl#StrictlyPositiveLowerBoundInd");
		}
		return false;
		
	}
	
	/**
	 * This method returns true iff the result block represented by the current instance of this class
	 * contains a IntegerData MathProp-Definition. Callers should now the context in order to conclude the right model entity with this property
	 * @param entid
	 * @return
	 */
	public Boolean containsIntegerDataMathPropforEntity(String entname, String entvarname){
		if( this.bindings.containsKey(entvarname) && (this.bindings.get(entvarname).equals(entname))){
			return this.bindings.containsValue("http://www.semanticweb.org/florianstapel/ontologies/2015/6/OM.MProps.owl#IntegralityPropertyInd");
		}
		return false;
	}
	
	/**
	 * This method returns true iff the result block represented by the current instance of this class
	 * contains a Binary Data MathProp-Definition. Callers should now the context in order to conclude the right model entity with this property
	 * @param entid
	 * @return
	 */
	public Boolean containsBinaryDataMathPropforEntity(String entname, String entvarname){
		if( this.bindings.containsKey(entvarname) && (this.bindings.get(entvarname).equals(entname))){
			return this.bindings.containsValue("http://www.semanticweb.org/florianstapel/ontologies/2015/6/OM.MProps.owl#BinaryPropertyInd");
		}
		return false;
		
	}

	/**
	 * This method returns the string name of a realted entity queried by the variable relentityvarname if the result block is for entname that relates to
	 * the returned individual, e.g., by definedOn or requires. 
	 * Otherwise, null will be returned
	 * @param entname
	 * @param relentityvarname
	 * @return
	 */
	public String getRelatedEntityforEntityName(String entname, String relentityvarname) {
		String result = this.bindings.get(relentityvarname);
		
		if(this.bindings.containsValue(entname)){return result;}
		else {return null;}
		
	}

	public boolean containsOntologyTypeInformationforEntity(String entname,
			String entvarname, String ontotypeiri) {
		if( this.bindings.containsKey(entvarname) && (this.bindings.get(entvarname).equals(entname))){
			return this.bindings.containsValue(ontotypeiri);
		}
		return false;
	}

	
	
}
