package de.dsor.ontooptmod.models;

import java.io.File;

import org.semanticweb.owlapi.*;
import org.semanticweb.owlapi.apibinding.OWLManager;
import org.semanticweb.owlapi.io.FileDocumentSource;
import org.semanticweb.owlapi.io.OWLOntologyDocumentSource;
import org.semanticweb.owlapi.io.StringDocumentSource;
import org.semanticweb.owlapi.model.IRI;
import org.semanticweb.owlapi.model.MissingImportHandlingStrategy;
import org.semanticweb.owlapi.model.OWLAnnotation;
import org.semanticweb.owlapi.model.OWLAnnotationProperty;
import org.semanticweb.owlapi.model.OWLClass;
import org.semanticweb.owlapi.model.OWLClassExpression;
import org.semanticweb.owlapi.model.OWLDataFactory;
import org.semanticweb.owlapi.model.OWLImportsDeclaration;
import org.semanticweb.owlapi.model.OWLIndividual;
import org.semanticweb.owlapi.model.OWLNamedIndividual;
import org.semanticweb.owlapi.model.OWLOntology;
import org.semanticweb.owlapi.model.OWLOntologyCreationException;
import org.semanticweb.owlapi.model.OWLOntologyLoaderConfiguration;
import org.semanticweb.owlapi.model.OWLOntologyManager;
import org.semanticweb.owlapi.model.OWLSubClassOfAxiom;
import org.semanticweb.owlapi.reasoner.InferenceType;
import org.semanticweb.owlapi.reasoner.NodeSet;
import org.semanticweb.owlapi.reasoner.OWLReasoner;
import org.semanticweb.owlapi.reasoner.OWLReasonerFactory;
import org.semanticweb.owlapi.reasoner.ReasonerInternalException;
import org.semanticweb.owlapi.reasoner.structural.StructuralReasonerFactory;
import org.semanticweb.owlapi.util.OWLOntologyMerger;

import java.util.*;

import de.dsor.ontooptmod.helpers.*;
import de.dsor.ontooptmod.derivation.*;

/**
 * This class is used to manage ontology represented optimization models. To this purpose, it manages an OWL API OWL Ontology Manager and several OWL API OWL Ontology Models.
 * @author Florian Stapel
 *
 */
public class OptimizationOntologyManager {

	private OWLOntologyManager manager;
	private OWLOntology optmodel;
	private OWLOntology mergedModel;
	private OWLDataFactory df;
	private String modelIRIBase;
	private Set<IRI> directImports;
	private Set<IRI> indirectImports;
	private OWLReasonerFactory rFact;
	private List<Pair<OWLClass, Pair<OWLAnnotation, OWLAnnotation>>> processedModelEntitieswithAnnotations;
	
	
	/**
	 * This constructor implementation loads an ontology model from a File by creating an OWLOntologyManager and multiple ontologies on this given by the parameter file.
	 * First the model from the file is loaded, afterwards all direct and indirect imports from the file system. A reasoner factory is initialized for later
	 * use by the methods of this class. The constructor also creates a merge of all loaded ontologies with a default IRI.
	 * @param file
	 */
	public OptimizationOntologyManager(File file) {
		manager = OWLManager.createOWLOntologyManager();
		df = OWLManager.getOWLDataFactory();
		
		// Try to use the method below, but adding IRI Mappers leads to an unpredictable behaviour: SILENTIMportHandling seems to ge ignored!
		//MyOwlOntologyImportHelper.addLocalVocabularyIRIMappers(manager);
		
		// Test wether OM IRI can be mapped: To Do, no methods provded by OWL API
		
		// Test Block: Just load a hard coded Ontology by IRI to see whether IRI Mappers will then work:	
		/*
		try {
			optmodel = manager.loadOntologyFromOntologyDocument(IRI.create("http://www.semanticweb.org/florianstapel/ontologies/2014/8/OM-Network-Constraints.owl"));
		} catch (OWLOntologyCreationException e1) {
			// TODO Auto-generated catch block
			e1.printStackTrace();
		}
		*/
		
		// Test Block: Create Dummy OM Ontology to test wether it's IRI can be mapped on file system
		/*
		try {
			OWLOntology testOM = manager.createOntology(IRI.create("http://www.semanticweb.org/florianstapel/ontologies/2014/8/OM.owl"));
			System.out.println(manager.getOntologyDocumentIRI(testOM));
			
		} catch (OWLOntologyCreationException e1) {
			// TODO Auto-generated catch block
			e1.printStackTrace();
		}
		*/
	
		try {
			// To Do: Put file into an Ontology Reader
			//optmodel =  manager.loadOntologyFromOntologyDocument(new FileDocumentSource(file), new OWLOntologyLoaderConfiguration().setMissingImportHandlingStrategy(MissingImportHandlingStrategy.SILENT));
			
			// Try 1 with a config and a file
			/*
			OWLOntologyLoaderConfiguration config = new OWLOntologyLoaderConfiguration();
			config.setMissingImportHandlingStrategy(MissingImportHandlingStrategy.THROW_EXCEPTION);
			optmodel = manager.loadOntologyFromOntologyDocument(new FileDocumentSource(file), config);
			*/
			
			// Try 2 with a StringDocumentSource and a config
			/*
			optmodel = manager.loadOntologyFromOntologyDocument(new StringDocumentSource(file.toString()));
			*/
			
			// Try 3 Convert the file to an IRI by getting its string representation and try to load that (hopefully with imports by mappers)
			/*
			System.out.println(IRI.create(file.toString()).toString());
			optmodel = manager.loadOntologyFromOntologyDocument(IRI.create(file.toString()));
			*/
			
			//Try 4 Convert the file to an IRI by geting its uri representation and try to load that ...
			// This seems to work until we come to the load of imports: IRI is resolved to a correct path but no suited "factory" is found
			/*
			System.out.println(IRI.create(file.toURI()).toString());
			optmodel = manager.loadOntologyFromOntologyDocument(IRI.create(file.toURI()));
			*/
			
			// Try 5: Load first form a file source with suppressed Missing Imports Exceptions. Then try to load imports manually and do not use IRI Mappers at all.
			
			optmodel =  manager.loadOntologyFromOntologyDocument(new FileDocumentSource(file), new OWLOntologyLoaderConfiguration().setMissingImportHandlingStrategy(MissingImportHandlingStrategy.SILENT));
			modelIRIBase = optmodel.getOntologyID().getOntologyIRI().toString();
			
			directImports = new TreeSet();
			System.out.println("---------------Here come the direct imports:---------------------------------------");
			
			for(OWLImportsDeclaration imp: optmodel.getImportsDeclarations()){
				System.out.println("Ontology IRI: " + imp.getIRI().toString() + " Document IRI: " + MyOwlOntologyImportHelper.getDocumentIRIforOntology(imp.getIRI()).toString());
	
					if(!manager.contains(imp.getIRI())){
						try{
						OWLOntology nextOnto = manager.loadOntologyFromOntologyDocument(new FileDocumentSource(new File(MyOwlOntologyImportHelper.getDocumentIRIforOntology(imp.getIRI()).toString())), new OWLOntologyLoaderConfiguration().setMissingImportHandlingStrategy(MissingImportHandlingStrategy.SILENT));
						directImports.add(nextOnto.getOntologyID().getOntologyIRI());
						}
						catch(OWLOntologyCreationException e){
							System.err.println("Error when loading a direct Ontology import" );
						}			
					}

			}
			
			// Next we try to also load the indirect imports by a self-written method
			indirectImports = new TreeSet();
			for(OWLImportsDeclaration imp: optmodel.getImportsDeclarations()){
				this.loadIndirectImports(imp);
			}
			
		} catch (OWLOntologyCreationException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
			System.err.println("Error when loading Ontology and it's imports from File and local File System" );
		}	
	
		// Now we create the reasoner and a factory for later use (Recall that even transitive subclass axioms would be missing without this)
		try{
		rFact = new StructuralReasonerFactory();
		}
		catch(ReasonerInternalException e){System.err.println("Something gone wrong when initializing the reasoner factory");}
		
		// We now try to merge all loaded ontologies into a new one
		mergedModel = this.mergeAllOntologiesInModelManager(IRI.create(modelIRIBase + "/allImportsMerged"));
		
	}
	
	/**
	 * Getter for current Opt Model
	 * @return
	 */
	public OWLOntology getoptModel(){
		return this.optmodel;
	}
	
	/**
	 * Getter for merge of all Ontologies in the current Opt Model
	 * @return
	 */
	public OWLOntology getmergedModel(){
		return this.mergedModel;
	}
	 
	
	/*
	 * This method tries to load indirect ontology imports by recursively adding the imports for a given import Ontology
	 * To do. Simplify code also in calling method to only one recursive call for all imports (indirect or not)
	 */
	private void loadIndirectImports(OWLImportsDeclaration imp) {
		try{
			for(OWLImportsDeclaration indimp: manager.getOntology(imp.getIRI()).getImportsDeclarations()){
				if(!manager.contains(indimp.getIRI())){
				OWLOntology actindimponto = manager.loadOntologyFromOntologyDocument(new FileDocumentSource(new File(MyOwlOntologyImportHelper.getDocumentIRIforOntology(indimp.getIRI()).toString())), new OWLOntologyLoaderConfiguration().setMissingImportHandlingStrategy(MissingImportHandlingStrategy.SILENT));
				indirectImports.add(actindimponto.getOntologyID().getOntologyIRI());
				if (!actindimponto.getImportsDeclarations().isEmpty()){ this.loadIndirectImports(indimp);}
				}
			}
		}
		catch(OWLOntologyCreationException e){System.err.println("Error when loading an indirect import");}
		
	}

/**
 * This method tries to return the IRI of the currently loaded optimization model as a string.
 * @return
 */
	public String getModelName(){
		String name = "none";
		try{
			name = optmodel.getOntologyID().getOntologyIRI().toString(); // Check: IRI vs. URI
		} catch (NullPointerException e) {
			System.err.println("No Ontology Model Loaded");
		}
		return name;
	}
	
	/**
	 * This method returns the direct Imports of the actual optimization Model as a Set of Strings
	 * @return
	 */
	public Collection<String> getModelDirectImportsAsStrings(){
		Collection<String> vocs = new LinkedList<String>();
			for( IRI iri: this.directImports ){
				vocs.add(iri.toString());									
			}
		return vocs;	
	}
	
	/**
	 * This method returns the indirect Imports of the actual optimization Model as a Set of Strings
	 * @return
	 */
	public Collection<String> getModelIndirectImportsAsStrings(){
		Collection<String> vocs = new LinkedList<String>();
			for( IRI iri: this.indirectImports ){
				vocs.add(iri.toString());									
			}
		return vocs;	
	}
	
	/*
	 * This method returns the data of the optimization model for a suitably formatted JTable
	 */
	public Object[][] retrieveEntityTableData() {
		
		// Get the data into a HashMap of Lists for each entity Type
		System.out.println("----------------Now trying to get a table of all model entities--------------------");
		
		XType [] entityTypesinOrder = {XType.Goal, XType.Constraint, XType.Set, XType.Variable, XType.Parameter};
		ArrayList<ArrayList<String>> tabledata = new ArrayList<ArrayList<String>>(); 
		
		for(XType entType : entityTypesinOrder){
			ArrayList<String> entityElements = new ArrayList<String>();
			for( OWLNamedIndividual ind: this.getXTypeEntities(mergedModel, entType)){
				entityElements.add(this.getPrefixFreeIRIasString(ind.getIRI()));
				System.out.println(ind.toStringID());
			}
			tabledata.add(entityElements);
		}
		System.out.println("----------------Got the table------------------------------------------------------");
		
		// Transform above's result data structure into a homogeneous non-sparse Array with empty strings by a helper method
		String [][] results = MyOptDataTransformHelper.getStringArrayfromArrayListofStringCollections(tabledata);
		
		return results;
	}
	
	/**
	 * This method returns the goal entities of the argument ontology as a Set of Strings
	 * @deprecated
	 * @param ontology
	 * @return
	 */
	@Deprecated private Set<OWLNamedIndividual> getGoals(OWLOntology ontology) {
		IRI goalIRI = IRI.create( "http://www.semanticweb.org/florianstapel/ontologies/2014/8/OM.owl#Goal");
		OWLClass goalClass = df.getOWLClass(goalIRI);
		
		System.out.println("--------------------------------------------------------------------------------------");
		System.out.println("Trying to extraxt all Goal individuals from the Ontology: " + ontology.getOntologyID().getOntologyIRI().toString());
		System.out.println("But first: The IRI of the Goal Concept: " + goalIRI.toString() );
		
		Set<OWLNamedIndividual> individuals = null;
		// Use the reasoner to get the individuals
		try{
			// First compute subclasses
			//reas.precomputeInferences(InferenceType.CLASS_HIERARCHY);
			// Now get the individuals under goal
			OWLReasoner reasnewonto = rFact.createReasoner(ontology);
			individuals = reasnewonto.getInstances(goalClass, false).getFlattened();
			System.out.println("Number of found Goal Individuals:" + individuals.size());

		}
		catch(ReasonerInternalException e){System.err.println("Exception when invoking reasoner to get goal individuals");}
		return individuals;
	}
	
	/**
	 * This method returns the xtype entities of the argument ontology as a set of Strings. The xtype is given as member of the enum XType
	 * to indicate wether we are looking for goals, constraints, sets, parameter or variables.
	 * @param ontology
	 * @param xtypeIRI
	 * @return
	 */
	private Set<OWLNamedIndividual> getXTypeEntities(OWLOntology ontology, XType xtype){
		IRI classIRI = null;
		
		switch(xtype){
			case Goal: classIRI = IRI.create("http://www.semanticweb.org/florianstapel/ontologies/2014/8/OM.owl#Goal");
					   break;
			case Constraint: classIRI = IRI.create("http://www.semanticweb.org/florianstapel/ontologies/2014/8/OM.owl#Constraint");
							 break;
			case Set: classIRI = IRI.create("http://www.semanticweb.org/florianstapel/ontologies/2014/8/OM.owl#Set");
			 	 	  break;
			case Variable: classIRI = IRI.create("http://www.semanticweb.org/florianstapel/ontologies/2014/8/OM.owl#Variable");
			 			   break;
			case Parameter: classIRI = IRI.create("http://www.semanticweb.org/florianstapel/ontologies/2014/8/OM.owl#Parameter");
			 				break;
			default: System.err.println("Mistake when initializing the xtype IRI for the getXTypeEntities Method. To Do: This should throw some kind of Exception");
					 break;	 
		}
		
		OWLClass entityClass = df.getOWLClass(classIRI);
		
		System.out.println("--------------------------------------------------------------------------------------");
		System.out.println("Trying to extraxt all " + xtype.toString() + " individuals from the Ontology: " + ontology.getOntologyID().getOntologyIRI().toString());
		System.out.println("But first: The IRI of the " + xtype.toString() + " Concept: " + classIRI.toString() );
		
		Set<OWLNamedIndividual> individuals = null;
		
		// Use the reasoner to get the individuals
		try{
			// First compute subclasses
			//reasnewonto.precomputeInferences(InferenceType.CLASS_HIERARCHY);
			// Now get the individuals under the entity class
			OWLReasoner reasnewonto = rFact.createReasoner(ontology);
			individuals = reasnewonto.getInstances(entityClass, false).getFlattened();
			System.out.println("Number of found Individuals:" + individuals.size());

		}
		catch(ReasonerInternalException e){System.err.println("Exception when invoking reasoner to get goal individuals");}
		return individuals;
		
	}
	
	
	/**
	 * Method used to print out all instances in the Ontology on the console. Can be used to test
	 * wether the reasoner works or indicate mistakes in the loaded ontology. By the argument onto the ontology to reason on can be specified.
	 * By a boolean Parameter you can specify whether or not to work on the imports closure. 
	 * Open Issue: How get the classes which are only used in axioms in the imported ontologies?
	 * @param onto
	 * @param workOnImportsClosure 
	 */
	public void allModelEntitiesOnConsole(OWLOntology onto, boolean workOnImportsClosure){
		
		OWLReasoner reasnewonto = rFact.createReasoner(onto);
		
		System.out.println("--------------------------------------------------------------------------------------");
		System.out.println("Trying to get a List of used classes and Individuals in Ontology: " + onto.getOntologyID().getOntologyIRI().toString() );
		System.out.println("Here's a List of all found model individuals and the type for which they where found:");
		for (OWLClass c: onto.getClassesInSignature(workOnImportsClosure)) {
			System.out.println("Individuals found for class " + c.getIRI().toString() + ":");
			for(OWLNamedIndividual i: reasnewonto.getInstances(c, true).getFlattened()){
			 System.out.println(i.getIRI().toString());	
			}
			System.out.println("..................next Class......................................................");
		}
		System.out.println("--------------------------------------------------------------------------------------");
	}
	
	/**
	 * This method tries to print out all known subclasses of a parameter IRI for a class on the console. The subclasses as well as the class itself should reside in the import closure of
	 * the ontology given by the first argument. Remark: This method may not be tested sufficiently!
	 * @param onto
	 * @param cls
	 */
	public void allSubClassesonConsole(OWLOntology onto, IRI clsIRI){
		OWLClass dummy = df.getOWLClass(clsIRI);
		System.out.println("--------------------------------------------------------------------------------------");
		System.out.println("Here's a List of all Subclass Axioms for " + clsIRI.toString() + " that were directly stated in the following ontology:");
		System.out.println("--- " + onto.getOntologyID().getOntologyIRI().toString() + ", let's go ----");
		
		for(OWLSubClassOfAxiom cls : onto.getSubClassAxiomsForSubClass(dummy)){
			System.out.println(cls.toString());
		}
		
		System.out.println("Here's another List of subClasses after invocation of a Reasoner:");
		
		// Use the reasoner to get the individuals
		try{
			// First compute subclasses
			OWLReasoner reasnew = rFact.createReasoner(onto);
			
			// To Do: Overwork Precomputation of Inferences
			reasnew.precomputeInferences(InferenceType.CLASS_HIERARCHY);
			// Now get the individuals under the input class
			for( OWLClass cls: reasnew.getSubClasses(dummy, false).getFlattened()){
			System.out.println(cls.getIRI().toString());
			}

		}
		catch(ReasonerInternalException e){System.err.println("Exception when invoking reasoner to get subclasses on console");}
		
		System.out.println("--------------------------------------------------------------------------------------");
	}
	
	/**
	 * This method tries to merge all the ontologies in the current OWLOntologyManager into a new Ontology with the IRI given by the input parameter.
	 * The merged ontology is added to the manager automatically and a reference comes as a return value.
	 * @param targetIRI
	 * @return
	 */
	private OWLOntology mergeAllOntologiesInModelManager(IRI targetIRI){
		System.out.println("--------------------------------------------------------------------------------------");
		System.out.println("I am currently trying to merge all Ontologies in the manager. The IRI of the merged Ontology will be:");
		System.out.println("*************" + targetIRI.toString() + "**********************");
		System.out.println("Here's a list of the merge input Ontologies:");
		for(OWLOntology onto: manager.getOntologies()){
			System.out.println(onto.getOntologyID().getOntologyIRI().toString());
		}
		System.out.println("--------------------------------------------------------------------------------------");
		
		OWLOntologyMerger merger = new OWLOntologyMerger(manager);
		OWLOntology merged = null;
		try {
			merged = merger.createMergedOntology(manager, targetIRI);
		} catch (OWLOntologyCreationException e) {
			System.err.println("Error when creating merged Ontology");
		}
		return merged;
		
	}
	
	/**
	 * This method simply tries to generate a String representation of an IRI element by extracting everything after a '#'. Use with care!
	 * @param iri
	 * @return
	 */
	private String getPrefixFreeIRIasString(IRI iri){
		String dummy = iri.toString();
		String result = "";
		try{
		result = dummy.substring(dummy.indexOf('#') + 1);
		}
		catch(IndexOutOfBoundsException e){System.err.println("Error when trying to generate a Prefix free substring for a IRI");}
		return result;
	}
	
	/**
	 * Return the current OWLOntologyManager
	 * @return
	 */
	public OWLOntologyManager getOWLOntologyManager() {
		return this.manager;
	}
	
	
	/**
	 * This method processes the current merged model to an AML Representation by iterating over the classes returned by 
	 * getProcessableEntityDocumentIRIsbyOWLClassKeysandQuery() and then processing the respective query and xslt documents
	 * obtained for the annotation value IRIs from a helper class.
	 * TODO: Ordered iteration for sets, parameters, variables, goals, constraints.
	 * @return TODO
	 */
	public String processCurrentModel2AML(){
		
		
		HashMap<OWLClass,Pair<OWLAnnotation, OWLAnnotation>> entityannotationsbyclass =  getProcessableEntityDocumentIRIsbyOWLClassKeysandQuery();
		List<Pair<OWLClass, Pair<OWLAnnotation, OWLAnnotation>>> entityannotationsinorder = this.sortClassKeyHashMapbyEntityTypes(entityannotationsbyclass);
		
		// Dirty but helps for quick implementation of generic validation
		this.processedModelEntitieswithAnnotations = entityannotationsinorder;
		
		MySPARQLDLQueryManager querymanager = new MySPARQLDLQueryManager(manager, mergedModel);
		StatementDerivationManager statman = new StatementDerivationManager();
		
		// deprecated code for iterating the unsorted HashMap
		/*
		for(OWLClass cls: entityannotationsbyclass.keySet()){
			System.out.println("Processing a Query, here's a little test of the current query IRI string: " + entityannotationsbyclass.get(cls).getFirst().getValue().toString());
			querymanager.processQueryFromFileLocation(MyOwlOntologyImportHelper.getFileLocationfromIRI(IRI.create(entityannotationsbyclass.get(cls).getFirst().getValue().toString()), DType.SPARQLDL_QUERY), MyOwlOntologyImportHelper.getFileLocationfromIRI(IRI.create(entityannotationsbyclass.get(cls).getFirst().getValue().toString()), DType.SPARQL_RESULT));
			statman.applyXSLTonXMLfromFiles(MyOwlOntologyImportHelper.getFileLocationfromIRI(IRI.create(entityannotationsbyclass.get(cls).getSecond().getValue().toString()), DType.XSLT), MyOwlOntologyImportHelper.getFileLocationfromIRI(IRI.create(entityannotationsbyclass.get(cls).getFirst().getValue().toString()), DType.SPARQL_RESULT), MyOwlOntologyImportHelper.getFileLocationfromIRI(IRI.create(entityannotationsbyclass.get(cls).getFirst().getValue().toString()), DType.STATEMENT_RESULT));
		}
		*/
		
		for(Pair<OWLClass, Pair<OWLAnnotation, OWLAnnotation>> cls: entityannotationsinorder){
			querymanager.processQueryFromFileLocation(MyOwlOntologyImportHelper.getFileLocationfromIRI(IRI.create(cls.getSecond().getFirst().getValue().toString()), DType.SPARQLDL_QUERY), MyOwlOntologyImportHelper.getFileLocationfromIRI(IRI.create(cls.getSecond().getFirst().getValue().toString()), DType.SPARQL_RESULT));
			//Old approach by calling XSLT (If reactivated choose method suffix byJAXP or bySAXON)
			//statman.applyXSLTonXMLfromFiles(MyOwlOntologyImportHelper.getFileLocationfromIRI(IRI.create(cls.getSecond().getSecond().getValue().toString()), DType.XSLT), MyOwlOntologyImportHelper.getFileLocationfromIRI(IRI.create(cls.getSecond().getFirst().getValue().toString()), DType.SPARQL_RESULT), MyOwlOntologyImportHelper.getFileLocationfromIRI(IRI.create(cls.getSecond().getFirst().getValue().toString()), DType.STATEMENT_RESULT));
			
			//We will now call directly with the Class IRI as a String and the respective OWL Annotations for query-result and the final statements
			//Generate the formulation Type-String by retireving everything behind the "#"-Separator:
			String formulationTypeString = cls.getFirst().getIRI().toString();
			int separatorIndex = formulationTypeString.indexOf('#');
			formulationTypeString = formulationTypeString.substring(separatorIndex + 1);
			System.out.println("Actual Class for Statement-Derivation without blank-space:" + formulationTypeString);
			//Call the Derivation Service
			statman.DelegateCalltoDerivationService(formulationTypeString, MyOwlOntologyImportHelper.getFileLocationfromIRI(IRI.create(cls.getSecond().getFirst().getValue().toString()), DType.SPARQL_RESULT), MyOwlOntologyImportHelper.getFileLocationfromIRI(IRI.create(cls.getSecond().getFirst().getValue().toString()), DType.STATEMENT_RESULT));
		}
		
		
		return statman.getCurrentAML();
	}
	
	/**
	 * This method simply reloads all XML Statements Files referenced by the class parts of the global list processedModelEntitieswithAnnotations
	 * and validates them against the AML XML Schmema
	 * @return
	 */
	public String validateCurrentAMLModelagainstSchema(){
		String result = "";
		XMLStatementManager statman = new XMLStatementManager(MyOwlOntologyImportHelper.getFileLocationfromString("AMPL-Statements-XML-Grammar", DType.XSD));
		
		for(Pair<OWLClass, Pair<OWLAnnotation, OWLAnnotation>> cls: processedModelEntitieswithAnnotations){
			statman.validateXMLFileagainstcurrentSchema(MyOwlOntologyImportHelper.getFileLocationfromIRI(IRI.create(cls.getSecond().getFirst().getValue().toString()), DType.STATEMENT_RESULT));
			result += "Validation for Individuals of  OWL Class " + this.getPrefixFreeIRIasString(cls.getFirst().getIRI()) + " : " + "Valid" + "\n";
		}
		
		return result;
	}
	
	/**
	 * This method sorts a HashMap of with OWLClass as Type in order of the five OM Ontology concepts set, param, variable, constraint, goal. 
	 * Therefore the input concepts must fall under that vocabulary. The return type is a List of key-value Pairs.
	 * Remark: Implementation can only work with string names of formulation types! It is not efficient to query individuals and OM-Types at this point!
	 * To that end we could do an explicit mapping of types to their order or we do this heuristically based on the containments of the type-strings.
	 * This is how it is done here!
	 * TODO: String based approach is not implemented correctly for variables!
	 */
	public <T> List<Pair<OWLClass,T>> sortClassKeyHashMapbyEntityTypes(HashMap<OWLClass,T> mapin){
		
		// Dirty solution: Sort the classes into 5 Lists for sets, params, vars.. and a default list
		// Later extensions can then also sort these buckets!
		
		List<Pair<OWLClass,T>> setlist = new ArrayList<Pair<OWLClass,T>>();
		List<Pair<OWLClass,T>> paramlist = new ArrayList<Pair<OWLClass,T>>();
		List<Pair<OWLClass,T>> varlist = new ArrayList<Pair<OWLClass,T>>();
		List<Pair<OWLClass,T>> goallist = new ArrayList<Pair<OWLClass,T>>();
		List<Pair<OWLClass,T>> constraintlist = new ArrayList<Pair<OWLClass,T>>();
		List<Pair<OWLClass,T>> defaultlist = new ArrayList<Pair<OWLClass,T>>();

		
		for(OWLClass cls: mapin.keySet()){
			Boolean inserted = false;
			
			String iristring = cls.getIRI().toString();
			
			if (!inserted && iristring.contains("Goal")){goallist.add(new Pair<OWLClass,T>(cls, mapin.get(cls))); inserted=true;}
			if (!inserted && (iristring.contains("Constraint") || iristring.contains("Bounds"))){constraintlist.add(new Pair<OWLClass,T>(cls, mapin.get(cls))); inserted=true;}
			if (!inserted && iristring.contains("Set")){setlist.add(new Pair<OWLClass,T>(cls, mapin.get(cls))); inserted=true;}
			if (!inserted && iristring.contains("Flow")){varlist.add(new Pair<OWLClass,T>(cls, mapin.get(cls))); inserted=true;}
			if (!inserted && ( iristring.contains("Collection")) || iristring.contains("Multiplier")){paramlist.add(new Pair<OWLClass,T>(cls, mapin.get(cls))); inserted=true;}
			
			
			if (!inserted ){defaultlist.add(new Pair<OWLClass,T>(cls, mapin.get(cls))); inserted=true; System.out.println("Added a default when sorting!");}
			inserted = false;	
		}
		
		List<Pair<OWLClass,T>> resultlist = new ArrayList<Pair<OWLClass,T>>();
		resultlist.addAll(setlist); resultlist.addAll(paramlist); resultlist.addAll(varlist); 
		resultlist.addAll(goallist); resultlist.addAll(constraintlist); resultlist.addAll(defaultlist);
		
		return resultlist;
	}

	
	/**
	 * This method returns the OWLAnnotations for all Documents that are required to process the model entities.
	 * The return value is a HashMap that has OWLClasses for the elements of the model that can be instantiated as keys. The value is a Pair (self-written class) of two OWLAnnotations.
	 * The first one being the DLQuery, the second one being the corresponding XSLT. The mapping is done by a helper method on string base!
	 * Note that the method works by returning the annotations of the documents 
	 * that are derived by first obtaining all model entities, then their classes and afterwards the annotation properties by iterating over the classes and possible properties. This assumes, that the classes
	 * and annotations are unique. Multiple individuals of the same class that result in multiple statements are in principle to be treated by the
	 * query, xslt and xsd implementations!
	 * @return
	 */
	public HashMap<OWLClass,Pair<OWLAnnotation, OWLAnnotation>> getProcessableEntityDocumentIRIsbyOWLClassKeysandQuery(){
		
		
		// First recompute all model entities
		System.out.println("----------------Trying to find out all model entities as individuals--------------------");
		
		XType [] entityTypesinOrder = {XType.Goal, XType.Constraint, XType.Set, XType.Variable, XType.Parameter};
		Collection<OWLNamedIndividual> entityIndividuals = new ArrayList<OWLNamedIndividual>();
		
		for(XType entType : entityTypesinOrder){
			for( OWLNamedIndividual ind: this.getXTypeEntities(mergedModel, entType)){
				entityIndividuals.add(ind);
				System.out.println(ind.toStringID());
			}
		}
		System.out.println("----------------Got the entities------------------------------------------------------");
		
		// Now extract all classes of the entities
		Set<OWLClass> entityClasses = new HashSet<OWLClass>();
		for(OWLNamedIndividual ind: entityIndividuals){
			Set<OWLClassExpression> indTypes = ind.getTypes(this.mergedModel);
			for(OWLClassExpression clsexpr: indTypes){
				OWLClass current = clsexpr.asOWLClass();
				entityClasses.add(current);
				System.out.println("----------------Found the following class: " + current.getIRI().toString() + "------------------------------------------------------");
			}

		}
		
		// Let us first get the annotation properties
		System.out.println("----------------Here come Document Annotation Properties------------------------------------------------------");
		Set<OWLAnnotationProperty> commentsubs = this.df.getRDFSComment().getSubProperties(mergedModel);
		for(OWLAnnotationProperty prop:commentsubs){
			System.out.println("Here's a sub annotation property of comment: " + prop.toString());
		}
		System.out.println("----------------That were Document Annotation Properties------------------------------------------------------");
		
		
		// Attention when iterating over the classes. The set of classes is too big and contains classes for which there are no annotation properties!
		System.out.println("----------------Looking for Document Annotations------------------------------------------------------");
		
		HashMap<OWLClass,Pair<OWLAnnotation, OWLAnnotation>> results = new HashMap<OWLClass,Pair<OWLAnnotation, OWLAnnotation>>();
		
		for(OWLClass cls:entityClasses){
			System.out.println("----------------Looking for Document Annotations of the class: " + cls.toString());
			
			Set<OWLAnnotation> annotations = new HashSet<OWLAnnotation>();
			for(OWLAnnotationProperty prop:commentsubs){
				Set<OWLAnnotation> newannotations = cls.getAnnotations(this.mergedModel, prop);
				if (!newannotations.isEmpty()) {annotations.addAll(newannotations);}
			}
			
			System.out.println("----------------I have found the following number of annotations for the class above: " + annotations.size());
			if(annotations.size() == 2) {results.put(cls, filterAndSortQueryandXSLAnnotations(annotations));}
		}

		System.out.println("----------------That were the Annotations------------------------------------------------------");
		
		return results;
	}
	
	
	/**
	 * This method in order extracts the unique OWLAnnotations for DLQueries and XSLTs.
	 * Attention: This method is very dirty and unsafe!
	 * It just works for Sets of Annotations that contain exactly the two annotation properties for DLQueries and XSLT Documents of the same Class!
	 * @param annotations
	 * @return
	 */
	private Pair<OWLAnnotation, OWLAnnotation> filterAndSortQueryandXSLAnnotations(
			Set<OWLAnnotation> annotations) {
		
		OWLAnnotation first = null;
		OWLAnnotation second = null;
		
		for(OWLAnnotation currentannotation: annotations){
			
				if(currentannotation.getValue().toString().contains("DLQuery")){
					first= currentannotation;
				}
				if(currentannotation.getValue().toString().contains("XSL")){
				    second=currentannotation;
				}
		}

		Pair<OWLAnnotation,OWLAnnotation> returnPair = new Pair<OWLAnnotation,OWLAnnotation>(first,second);
		System.out.println("----I have found the following pair of values for DLQuery and XSLT annotations: " + first.getValue().toString() + "," + second.getValue().toString());
		
		return returnPair;
	}

	/**
	 * This method returns the OWLAnnotations for all Documents that are required to process the model entities.
	 * The return HashMap uses the IRIs of the annotation properties as a key for which they store the Collection of annotations as a value.
	 * The typical cases are DL Queries and AML Statement. Be carefull that these annotation properties both exist in constraint and meta-domain ontologies!
	 * Note that the method works by returning the annotations of the documents 
	 * that are derived by first obtaining all model entities, then their classes and afterwards the annotation properties. This assumes, that the classes
	 * and annotations are unique. Multiple individuals of the same class that result in multiple statements are in principle to be treated by the
	 * query, xslt and xsd implementations!
	 * @return
	 */
	@Deprecated public HashMap<IRI,HashSet<OWLAnnotation>> getProcessableEntityDocumentIRIsbyPropertyIRIs(){
		
		
				// First recompute all model entities
				System.out.println("----------------Trying to find out all model entities as individuals--------------------");
				
				XType [] entityTypesinOrder = {XType.Goal, XType.Constraint, XType.Set, XType.Variable, XType.Parameter};
				Collection<OWLNamedIndividual> entityIndividuals = new ArrayList<OWLNamedIndividual>();
				
				for(XType entType : entityTypesinOrder){
					for( OWLNamedIndividual ind: this.getXTypeEntities(mergedModel, entType)){
						entityIndividuals.add(ind);
						System.out.println(ind.toStringID());
					}
				}
				System.out.println("----------------Got the entities------------------------------------------------------");
				
				// Now extract all classes of the entities
				Set<OWLClass> entityClasses = new HashSet<OWLClass>();
				for(OWLNamedIndividual ind: entityIndividuals){
					Set<OWLClassExpression> indTypes = ind.getTypes(this.mergedModel);
					for(OWLClassExpression clsexpr: indTypes){
						OWLClass current =clsexpr.asOWLClass();
						entityClasses.add(current);
						System.out.println("----------------Found the following class: " + current.getIRI().toString() + "------------------------------------------------------");
					}

				}
				
				System.out.println("----------------Looking for Document Annotations------------------------------------------------------");
				// Attention: The last step may have found multiple classes that are not used for instantiation
				// Now let's get all the annotation properties
				// TODO: Get those that are comments or better the subtype indicated by the enum
				
				// Let us first get the annotation properties
				System.out.println("----------------Here come Document Annotation Properties------------------------------------------------------");
				Set<OWLAnnotationProperty> commentsubs = this.df.getRDFSComment().getSubProperties(mergedModel);
				for(OWLAnnotationProperty prop:commentsubs){
					System.out.println("Here's a sub annotation property of comment: " + prop.toString());
				}
				System.out.println("----------------That were Document Annotation Properties------------------------------------------------------");
				System.out.println("----------------Here come the Annotations---------------------------------------------------------------------");
				// Trying to give out all values for the subproperties
				
				HashMap<IRI, HashSet<OWLAnnotation>> annotations = new HashMap<IRI,HashSet<OWLAnnotation>>();
				for(OWLAnnotationProperty prop:commentsubs){
					HashSet<OWLAnnotation> currentannotations = new HashSet<OWLAnnotation>();
					
					System.out.println("Here come the annotation values for " + prop.toString() + ":");
					for(OWLClass cls: entityClasses ){
						currentannotations.addAll(cls.getAnnotations(mergedModel, prop));
						}
					
					for(OWLAnnotation annot:currentannotations){
						System.out.println("value: " + annot.getValue().toString());
					}
					annotations.put(prop.getIRI(), currentannotations);
				}
			
				
				System.out.println("----------------That were the Annotations------------------------------------------------------");
				
		return annotations;
	}

}
