
set Nodes;
set JunctionNodes within Nodes;
set Sources within Nodes;
set Tanks within Nodes;
set Pipes within {Nodes , Nodes};
set BuildPipes within Pipes;            #Set of Pipes which are to be renewed within an certain intertal of Timesteps

param T > 0;							#Number of Timesteps for planning of building measures. Time counted from 0..T
param RenewTimeStart{BuildPipes} > 0 ;	#Time-Limit from which a Pipe can be renewed. Parameter is less equal T 
param RenewTimeEnd{BuildPipes} > 0   ;  #Time-Limit when a Pipe has to be renewed. Parameter is less equal T 
param Budget{1..T} ;					#Budget at Time t

param Per > 0; 							#Number of Periods for consideration of operational aspects like tank-filling. 
param deltaPer;							#Uniform length of a Period in Seconds [s]
param RTank{Tanks};						#Radius of Tanks in meters [m]
param HInit{Tanks,0..T};				#Initial Head, i.e., Filling of Tank at time T [m]

param dnum;								#Number of available Pipe Diameters
param degP;								#Degree of fitted cost-function polynomial
param deltaHeadloss;					#delta indicates the distance to the origin for which the Headloss-equation (DW or HW) is smoothed
param PI;								#Pi
param g;								#Earth gravitation [m/s^2]
param M;								#BigM

param Diameters{1..dnum};				#Set of commercially available Pipe Diameters as a vector of parameters [m]
param pcoeff{0..degP};					#Coefficients of fitted Cost-function Polynomial
param Ck{Pipes};						#Pipe roughness Coefficient depending on wheather HW (C) or DW (k) is used. HW can be implemented in SI Units, see english wiki- Here DW, Ck in [m]
param L{Pipes};							#Length of a Pipe [m]
param Demand{JunctionNodes,0..T,1..Per};#Demand at a Junction Node [m^3/s], time-dependent, Period-dependent: Demand is assumed to be constant in Period per.

# Attention: Minimal Pipe Velocities will be controlled in a Simulation
#param vmin{Pipes};						#Minimimal Pipe Velocities [m/s]  
param vmax{Pipes};						#Maximal Pipe Velocities   [m/s]
param dmin{BuildPipes};						#Minimal Pipe Diameter     [m]
param dmax{BuildPipes};						#Maximal Pipe Diameter     [m]
param dactual{Pipes};					#Actual Diameter of Pipe in the existing network at time 0 [m]
param QSmin{Sources};					#Minimal Source Inflow     [m]
param QSmax{Sources};					#Maximal Source Inflow     [m]

param Hmin{JunctionNodes union Tanks};	#Minimal Head of Junction Node/Tank [m]
param Hmax{JunctionNodes union Tanks};	#Maximal Head of Junction Node/Tank [m]
param HSources{Sources};				#Initial Head at Source Node [m]


var H{Nodes,0..T,1..Per} >=0;			#Head at Nodes at time t after Period per [m]
var Q{Pipes,0..T,1..Per};				#Flow through Pipes at time t, constant in Period per [m^3/s]. Later: Flow through Pumps
var r{Pipes,0..T} >=0;					#Resistance Coefficient of Pipes at time t. r is the same for all Periods of a time-step. Convention: Leave r as a Variale for all Pipes for simplicity. The Preprocessor will then compute the values and fix the respective variables
var d{BuildPipes,0..T}	;					#Variable Diameter of Pipes at time t [m]. Diameter is only Variable for BuildPipes. 
var X{BuildPipes,0..(dnum-1),0..T} binary;	#Pipe Diameter increment switching at time t [-]
var cost{BuildPipes,1..T} >= 0;					#Cost of building measures of Pipe (i,j) at time t [?]
var ncost{BuildPipes,1..T} binary;			#Binary indicating wether building cost for pipe (i,j) is counted at time t

#Objective
minimize GesamtKosten: sum{(i,j) in BuildPipes, t in 1..T}(L[i,j] * cost[i,j,t]);

#Budget-Restrictions: Definition of cost-variables in Pipe (i,j) at time t! Trying to avoid nonlinear multiplication
s.t. KostenAnijtequalspos{(i,j) in BuildPipes, t in 1..T}: cost[i,j,t] - sum{e in 0..degP}(pcoeff[e]*d[i,j,t]^e) <= M * ( 1 - ncost[i,j,t]);
s.t. KostenAnijtequalsneg{(i,j) in BuildPipes, t in 1..T}: (-1) * cost[i,j,t] + sum{e in 0..degP}(pcoeff[e]*d[i,j,t]^e) <= M * ( 1 - ncost[i,j,t]);
s.t. KostenAnijtswitch{(i,j) in BuildPipes, t in 1..T}: cost[i,j,t] <= M * ncost[i,j,t];

#Budget-Restrictions: Stay in Budget for every Timestep:
s.t. BudgetRestrictions{Tstar in 1..T}: sum{(i,j) in BuildPipes, t in 1..Tstar}(L[i,j] * cost[i,j,t]) <= sum{t in 1..Tstar}(Budget[t]);

#Mass-Balance at Junctions.
s.t. Massbilance{x in JunctionNodes, t in 0..T, per in 1..Per}: sum{(i,x) in Pipes}(Q[i,x,t,per]) - sum{(x,j) in Pipes}(Q[x,j,t,per]) = Demand[x,t,per];

#Mass-Balance for Tanks: Initial level for Tanks gives extra Restriction.
s.t. MassbilanceTankFirstPeriod{x in Tanks, t in 0..T}: sum{(i,x) in Pipes}(Q[i,x,t,1]) - sum{(x,j) in Pipes}(Q[x,j,t,1]) = ( PI * RTank[x]^2 /  deltaPer ) * (H[x,t,1] - HInit[x,t]) ;
s.t. MassbilanceTankLaterPeriods{x in Tanks, t in 0..T, per in 2..Per}: sum{(i,x) in Pipes}(Q[i,x,t,per]) - sum{(x,j) in Pipes}(Q[x,j,t,per]) = ( PI * RTank[x]^2 /  deltaPer ) * (H[x,t,per] - H[x,t,per-1]) ;


#Um Null geglaettete Druckverlustgleichung nach Darcy-Weisbach. 
#Kurve in [-delta,delta] ergibt sich aus einfachen Bedingungen 0.- 2. Ordnung an den Polynomansatz g(x) = ax + bx^3 + cx^5 fuerr f(x) = x * |x|
#Anmerkung: Fr diese Experimente wurden vereinfachte Koeffizienten verwendet, die die Bedingungen erster Ordnung auslassen. So entsteht ein Polynom vom Grad 3.
#Bei den richtigen Koeffizienten handelt es sich um a = 3/8 deltaHeadloss , b = 4,5 / (6 deltaHeadloss) , c = (- 1 / (8 deltaHeadloss ^ 3))
s.t. HeadlossDW{(i,j) in Pipes, t in 0..T, per in 1..Per}: (if abs(Q[i,j,t,per]) > deltaHeadloss then r[i,j,t] * Q[i,j,t,per] * abs(Q[i,j,t,per]) 
else r[i,j,t] * (2/3 * deltaHeadloss * Q[i,j,t,per] + 1/(3 * deltaHeadloss) * Q[i,j,t,per]^3)) = H[i,t,per] - H[j,t,per];


# Constant initial Heads at Sources
s.t. SourceHeadsConstant{s in Sources, t in 0..T, per in 1..Per}: H[s,t,per] = HSources[s];

# Bounds on Heads at Junction Nodes and Tanks
s.t. HeadBoundslow{n in JunctionNodes union Tanks, t in 0..T, per in 1..Per}: Hmin[n] <= H[n,t,per]; 
s.t. HeadBoundsup{n in JunctionNodes union Tanks, t in 0..T, per in 1..Per} : H[n,t,per]  <= Hmax[n];

# Restriktion fuer die Berechnung des "Resistance Coefficient". log10 ist der Logarithmus zur Basis 10 der in der Ingenieursliteratur mit lg oder log abgekuerzt wird.
# Unterscheide Restriktion nach "fixen" Pipes und BuildPipes um den Preprocessor die Werte f?r die fixen Pipes fixieren zu lassen
s.t. ResistanceCoefficientBuildPipes{(i,j) in BuildPipes, t in 0..T}: r[i,j,t] = (8 * L[i,j]) / (PI^2 * g * (d[i,j,t])^5) * (2 * log10((Ck[i,j]/ d[i,j,t] ) / 3.71))^(-2);
s.t. ResistanceCoefficient{(i,j) in (Pipes diff BuildPipes), t in 0..T}: r[i,j,t] = (8 * L[i,j]) / (PI^2 * g * (dactual[i,j])^5) * (2 * log10((Ck[i,j]/ dactual[i,j] ) / 3.71))^(-2);

# Upper Bounds for PipeSpeed. Distinction for Buildpipes and fixed Pipes!
s.t. PipeSpeedBoundsupPosBP{(i,j) in BuildPipes, t in 0..T, per in 1..Per}:       Q[i,j,t,per] <= vmax[i,j] * (PI/4) * d[i,j,t]^2;
s.t. PipeSpeedBoundsupNegBP{(i,j) in BuildPipes, t in 0..T, per in 1..Per}: (-1) *Q[i,j,t,per] <= vmax[i,j] * (PI/4) * d[i,j,t]^2;
s.t. PipeSpeedBoundsupPosFP{(i,j) in (Pipes diff BuildPipes), t in 0..T, per in 1..Per}:       Q[i,j,t,per] <= vmax[i,j] * (PI/4) * dactual[i,j]^2;
s.t. PipeSpeedBoundsupNegFP{(i,j) in (Pipes diff BuildPipes), t in 0..T, per in 1..Per}: (-1) *Q[i,j,t,per] <= vmax[i,j] * (PI/4) * dactual[i,j]^2;

#Pipe gets old and needs to be renewed in a certain time-limit
s.t. RenewinInterval{(i,j) in BuildPipes}: sum{t in RenewTimeStart[i,j]..RenewTimeEnd[i,j]}(ncost[i,j,t]) = 1;

# Coupling of Diameters for BuildPipes over time:

#Initial values:
s.t. Initialdiameters{(i,j) in BuildPipes}: d[i,j,0] = dactual[i,j];

#Coupling over time: 
s.t. CouplingTimeone{(i,j) in BuildPipes, t in 1..T}: ( d[i,j,t] - d[i,j,t-1] ) / dmax[i,j]   <= ncost[i,j,t] ;
s.t. CouplingTimetwo{(i,j) in BuildPipes, t in 1..T}: ( d[i,j,t-1] - d[i,j,t] ) / dmax[i,j]   <= ncost[i,j,t] ;

# Lower and Upper Bounds on the Outflow of Sources
s.t. SourceFlowBoundslow{s in Sources, t in 0..T, per in 1..Per}: QSmin[s] <= sum{(s,j) in Pipes}(Q[s,j,t,per]); 
s.t. SourceFlowBoundsup{s in Sources, t in 0..T, per in 1..Per} : sum{(s,j) in Pipes}(Q[s,j,t,per]) <= QSmax[s];

#All Outflows nonnegative by a set-intersection 
s.t. SourceFlownonneg{(s,j) in ({Sources,Nodes} inter Pipes), t in 0..T, per in 1..Per}: Q[s,j,t,per] >= 0;

#Lower and Upper Bounds on the Area/Diameter of BuildPipes
s.t. PipeDiameterBoundslow{(i,j) in BuildPipes, t in 0..T}: dmin[i,j] <= d[i,j,t];
s.t. PipeDiameterBoundsup{(i,j) in BuildPipes, t in 0..T}:  d[i,j,t] <= dmax[i,j];

# Represent discrete set of commercially available Pipe Diameters by increments (two constraints)
s.t. PipeDiametersincremental{(i,j) in BuildPipes, t in 0..T}: d[i,j,t] =  Diameters[1] + sum{ mu in 2..dnum}((Diameters[mu] - Diameters[mu-1])* X[i,j,mu-1,t]); 
s.t. Incrementalsalloneupto{(i,j) in BuildPipes, mu in 1..(dnum-2), t in 0..T}: X[i,j,mu,t] >= X[i,j,mu+1,t]; 
