#Basic water network case study model as a variant of Bragalli et al.

set Nodes;
set JunctionNodes within Nodes;
set Sources within Nodes;
set Pipes within {Nodes , Nodes};
param dnum;		#Number of available Pipe Diameters
param degP;		#Degree of fitted cost-function polynomial
param PI;		#Pi
param g;		#Earth gravitation [m/s^2]
param Diameters{1..dnum};	#Set of commercially available Pipe Diameters in[m]
param pcoeff{0..degP};		#Coefficients of fitted Cost-function Polynomial
param Ck{Pipes};			#Pipe roughness Coefficient in [m]
param L{Pipes};				#Length of a Pipe [m]
param Demand{JunctionNodes} ; #Demand at a Junction Node [m^3/s]
param vmax{Pipes};			  #Maximal Pipe Velocities   [m/s]
param dmin{Pipes};			  #Minimal Pipe Diameter     [m]
param dmax{Pipes};			  #Maximal Pipe Diameter     [m]
param QSmin{Sources};		  #Minimal Source Inflow     [m^3/s]
param QSmax{Sources};		  #Maximal Source Inflow     [m^3/s]
param Hmin{JunctionNodes};	  #Minimal Head of Junction Node [m]
param Hmax{JunctionNodes};	  #Maximal Head of Junction Node [m]
param HSources{Sources};	  #Initial Head at Source Node [m]

var H{Nodes} >=0;	 #Head at Nodes  [m]
var Q{Pipes};		 #Flow through Pipes [m^3/s]
var r{Pipes} >=0;	 #Resistance Coefficient of Pipes for DW
var d{Pipes}	;	 #Variable Diameter of Pipes [m]. 
var X{Pipes,0..(dnum-1)} binary;  #Pipe Diameter increment switching

#Objective
minimize Costs: sum{(i,j) in Pipes}(L[i,j] * sum{e in 0..degP}(pcoeff[e]*d[i,j]^e));

#Mass-Balance at Junctions. 
s.t. Massbalance{x in JunctionNodes}: sum{(i,x) in Pipes}(Q[i,x]) - sum{(x,j) in Pipes}(Q[x,j]) = Demand[x];

#Standard Headloss, unsmoothed
HeadlossDW{(i,j) in Pipes}: H[i] - H[j] = r[i,j] * Q[i,j] * abs(Q[i,j]);

# Restriction for Computation of "Resistance Coefficient". log10 has Basis 10
s.t. ResistanceCoefficientPipes{(i,j) in Pipes}: r[i,j] = (8 * L[i,j]) / (PI^2 * g * (d[i,j])^5) * (2 * log10((Ck[i,j]/ d[i,j] ) / 3.71))^(-2);

# Upper Bounds for PipeSpeed
s.t. PipeSpeedBoundsupPos{(i,j) in Pipes}:       Q[i,j] <= vmax[i,j] * (PI/4) * d[i,j]^2;
s.t. PipeSpeedBoundsupNeg{(i,j) in Pipes}: (-1) *Q[i,j] <= vmax[i,j] * (PI/4) * d[i,j]^2;

# Constant initial Heads at Sources
s.t. SourceHeadsConstant{s in Sources}: H[s] = HSources[s];
# Lower and Upper Bounds on the Outflow of Sources
s.t. SourceFlowBoundslow{s in Sources}: QSmin[s] <= sum{(s,j) in Pipes}(Q[s,j]); 
s.t. SourceFlowBoundsup{s in Sources} : sum{(s,j) in Pipes}(Q[s,j]) <= QSmax[s];
#All Outflows nonnegative by a set-intersection 
s.t. SourceFlowNonneg{(s,j) in ({Sources,Nodes} inter Pipes)}: Q[s,j] >= 0;

# Bounds on Heads at Junction Nodes
s.t. HeadBoundslow{n in JunctionNodes}: Hmin[n] <= H[n]; 
s.t. HeadBoundsup{n in JunctionNodes} : H[n]  <= Hmax[n];

#Lower and Upper Bounds on the Diameters
s.t. PipeDiameterBoundslow{(i,j) in Pipes}: dmin[i,j] <= d[i,j];
s.t. PipeDiameterBoundsup{(i,j) in Pipes}:  d[i,j] <= dmax[i,j];

# Discrete set of commercially available Pipe Diameters by increments
s.t. PipeDiametersincremental{(i,j) in Pipes}: d[i,j] =  Diameters[1] + sum{ mu in 2..dnum}((Diameters[mu] - Diameters[mu-1])* X[i,j,mu-1]); 
s.t. Incrementalsalloneupto{(i,j) in Pipes, mu in 1..(dnum-2)}: X[i,j,mu] >= X[i,j,mu+1]; 
